#ifndef _HAL_VARAN_CLIENT_H
#define _HAL_VARAN_CLIENT_H

/* Includes */
#include "rX_Compiler.h"
#include "varan_client_proc_xpec_regdef.h"
#include "varan_client_sync_xpec_regdef.h"

/* Definitions */
#define VARAN_CLIENT_MEM_BUF_SIZE 128 /* maximum memory buffer size */
#define VARAN_CLIENT_OUT_MEM_CNT    2 /* number of output memory areas */
#define VARAN_CLIENT_IN_MEM_CNT     2 /* number of input memory areas */

#define VARAN_CLIENT_PROC_PORT 0
#define VARAN_CLIENT_SYNC_PORT 1

#define SPI_CMD_PAGE_READ    1
#define SPI_CMD_PAGE_WRITE   2
#define SPI_CMD_SECTOR_ERASE 3

#define ETH_FIFO_EMPTY 0 /**< Empty pointer FIFO */
#define ETH_FIFO_IND   1 /**< Indication FIFO    */
#define ETH_FIFO_REQ   2 /**< Request FIFO       */
#define ETH_FIFO_CON   3 /**< Confirmation FIFO  */

/* MAC address structure */
typedef uint8_t ETHERNET_MAC_ADDR_T[6];

/*****************************************************************************/
/*! Ethernet Frame Structure                                                 */
/*****************************************************************************/
typedef __PACKED_PRE struct ETHERNET_FRAME_Ttag
{
  uint32_t             ulDataLen;       /**< Frame data length including DA, SA, LT, Data, excluding FCS, PREAMBLE, SFD */
  ETHERNET_MAC_ADDR_T  tDstAddr;        /**< Destination MAC address (DA)     */
  ETHERNET_MAC_ADDR_T  tSrcAddr;        /**< Source MAC address (SA)          */
  uint16_t             usType;          /**< Frame length/type (LT)           */
  uint8_t              abData[1504];    /**< Frame data excluding DA, SA, LT, FCS */
  uint8_t              abRes[14];       /**< reserved, shall be set to zero   */
} __PACKED_POST ETHERNET_FRAME_T;

/*****************************************************************************/
/*! IP Fragment Reception Structure                                          */
/*****************************************************************************/
typedef __PACKED_PRE struct RX_IP_FRAGMENT_Ttag
{
  volatile uint8_t  bTypeStat;       /**< Fragment Type and Status */
  volatile uint8_t  abData[129];     /**< Fragment data */
  volatile uint16_t usLength;        /**< Fragment length including TypeStat  */
} __PACKED_POST RX_IP_FRAGMENT_T;

/*****************************************************************************/
/*! IP Fragment Transmission Structure                                       */
/*****************************************************************************/
typedef __PACKED_PRE struct TX_IP_FRAGMENT_Ttag
{
  volatile uint16_t usLength;        /**< Fragment length including Res/Type/Stat */
  volatile uint8_t  bRes;            /**< reserved, shall be set to zero          */
  volatile uint8_t  bTypeStat;       /**< Fragment Type and Status                */
  volatile uint8_t  abData[128];     /**< Fragment data                           */
} __PACKED_POST TX_IP_FRAGMENT_T;

#define MSK_FOREIGN_RESPONSE_TYPE_STATUS_TYPE        0x3f
#define SRT_FOREIGN_RESPONSE_TYPE_STATUS_TYPE           0
#define VAL_FOREIGN_RESPONSE_TYPE_STATUS_TYPE_DUMMY     0
#define VAL_FOREIGN_RESPONSE_TYPE_STATUS_TYPE_IP        1
#define VAL_FOREIGN_RESPONSE_TYPE_STATUS_TYPE_SAFETY    2
#define MSK_FOREIGN_RESPONSE_TYPE_STATUS_FIFO_EMPTY  0x40
#define SRT_FOREIGN_RESPONSE_TYPE_STATUS_FIFO_EMPTY     6
#define MSK_FOREIGN_RESPONSE_TYPE_STATUS_COMPLETE    0x80
#define SRT_FOREIGN_RESPONSE_TYPE_STATUS_COMPLETE       7

/*****************************************************************************/
/*! Result Codes for Functions                                               */
/*****************************************************************************/
typedef enum VRNCLNT_RESULTtag
{
  VRNCLNT_OKAY,                   /**< Successful                                   */
  VRNCLNT_ERR_INIT_FAILED,        /**< Initialization has failed                    */
  VRNCLNT_ERR_INVAL_PARAM,        /**< Invalid parameter                            */
  VRNCLNT_NO_DATA_AVAILABLE,      /**< No new output data available                 */
  VRNCLNT_EMAC_NO_DATA_AVAILABLE, /**< No new Ethernet indication available         */
  VRNCLNT_EMAC_FIFO_ERR,          /**< FIFO Error                                   */
  VRNCLNT_EMAC_SFD_ERR,           /**< SFD not found                                */
  VRNCLNT_EMAC_LEN_ERR,           /**< Ethernet length error                        */
  VRNCLNT_EMAC_FCS_ERR            /**< Ethernet FCS error                           */
} VRNCLNT_RESULT;

/*****************************************************************************/
/*! Sync Out IDs                                                             */
/*****************************************************************************/
typedef enum VARAN_SYNC_OUT_IDtag
{
  SYNC_OUT_ID_UNUSED,             /**< This output is not available                 */
  SYNC_OUT_ID_PLL_IRQ,            /**< PLL IRQ OS trigger                           */
  SYNC_OUT_ID_VARAN,              /**< VARAN VARAN Manager trigger                  */
  SYNC_OUT_ID_TI,                 /**< Time for data in valid                       */
  SYNC_OUT_ID_TO,                 /**< Time for data out valid                      */
  SYNC_OUT_ID_TIO                 /**< Time for data in/out valid                   */
} VARAN_SYNC_OUT_ID;

/*********************************************/
/*! Structure Memory Parameter Set           */
/*********************************************/
typedef struct VRNCLNT_MEM_PARAM_Ttag
{
  uint16_t usOffset; /**< Offset memory area, ignored if bSize == 0 */
  uint8_t  bSize;    /**< Size memory area, shall be between 1..128, 0: memory area unused */
} VRNCLNT_MEM_PARAM_T;

/*********************************************/
/*! Structure for initialize Parameter Set   */
/*********************************************/
typedef struct VRNCLNT_INIT_PARAMETER_SETtag
{
  VRNCLNT_MEM_PARAM_T atMemWr[VARAN_CLIENT_OUT_MEM_CNT]; /**< Output memory parameters, VARAN Manager writes, CPU reads */
  VRNCLNT_MEM_PARAM_T atMemRd[VARAN_CLIENT_IN_MEM_CNT];  /**< Input memory parameters, VARAN Manager reads, CPU writes */

  int                fIrqEnWdgExp;                   /**< WatchDog expired interrupt enable                    */
  int                fIrqEnGlobRst;                  /**< Global reset interrupt enable                        */
  int                fIrqEnCfgClr;                   /**< Configuration clear interrupt enable                 */
  int                fIrqEnEthFrameXmit;             /**< Ethernet frame transmitted interrupt enable          */
  int                fIrqEnEthFrameRcvd;             /**< Ethernet frame received interrupt enable             */
  int                fIrqEnSpiReq;                   /**< SPI request interrupt enable                         */
  int                fIrqEnPllLocked;                /**< PLL locked interrupt enable                          */
  int                fIrqEnPllUnlocked;              /**< PLL unlocked interrupt enable                        */
  int                fIrqEnSyncOut0;                 /**< Sync out 0 interrupt enable                          */
  int                fIrqEnSyncOut1;                 /**< Sync out 1 interrupt enable                          */
  int                fEmacEnable;                    /**< Ethernet MAC functionality enable                    */

  uint16_t           usPllP;                         /**< PLL control P factor                                 */
  uint16_t           usPllI;                         /**< PLL control I factor                                 */
  uint32_t           ulDeviceDelayCorrection;        /**< Device Delay Correction in relation to FPGA          */
  int                fOutputDriverSyncOut0;          /**< Sync out 0 output driver, 1/0: open-drain/push-pull, Note: Open-Drain not supported */
  int                fPolaritySyncOut0;              /**< Sync out 0 polarity, 1/0: high/low active            */
  int                fOutputEnableSyncOut0;          /**< Sync out 0 output enable, 1/0: enabled/disabled      */
  VARAN_SYNC_OUT_ID  eSyncOutID0;                    /**< Sync out 0 ID                                        */
  int                fOutputDriverSyncOut1;          /**< Sync out 1 output driver, 1/0: open-drain/push-pull, Note: Open-Drain not supported */
  int                fPolaritySyncOut1;              /**< Sync out 1 polarity, 1/0: high/low active            */
  int                fOutputEnableSyncOut1;          /**< Sync out 1 output enable, 1/0: enabled/disabled      */
  VARAN_SYNC_OUT_ID  eSyncOutID1;                    /**< Sync out 1 ID                                        */
  uint32_t           ulPulseLenSyncOut;              /**< Sync out pulse length [10 ns]                        */

  uint8_t            bPhyAddrVaranIn;                /**< PHY address of VARAN In (UpLink) port, shall be between 0..31 */
  uint8_t            bPhyAddrVaranOut;               /**< PHY address of VARAN Out port, shall be between 0..31         */

} VRNCLNT_INIT_PARAMETER_SET;

/***********************************************************************************************/
/*! Parameters ID that can be set/get by VaranClient_SetParameter()/VaranClient_GetParameter() */
/***********************************************************************************************/
typedef enum VRNCLNT_PARAMETERtag
{
  VRNCLNT_PARAMETER_RESERVED,                    /**< Reserved                                             */
  VRNCLNT_PARAMETER_WDG_TIME,                    /**< WatchDog time in 10ns resolution                     */
  VRNCLNT_PARAMETER_DEV_BASE_ADDR,               /**< Device Base Address (read only)                      */
  VRNCLNT_PARAMETER_MEM1_OFFSET_WRITE,           /**< Offset memory write area 1 (read only)               */
  VRNCLNT_PARAMETER_MEM1_SIZE_WRITE,             /**< Size memory write area 1 (read only)                 */
  VRNCLNT_PARAMETER_MEM1_OFFSET_READ,            /**< Offset memory read area 1 (read only)                */
  VRNCLNT_PARAMETER_MEM1_SIZE_READ,              /**< Size memory read area 1 (read only)                  */
  VRNCLNT_PARAMETER_MEM2_OFFSET_WRITE,           /**< Offset memory write area 2 (read only)               */
  VRNCLNT_PARAMETER_MEM2_SIZE_WRITE,             /**< Size memory write area 2 (read only)                 */
  VRNCLNT_PARAMETER_MEM2_OFFSET_READ,            /**< Offset memory read area 2 (read only)                */
  VRNCLNT_PARAMETER_MEM2_SIZE_READ,              /**< Size memory read area 2 (read only)                  */
  VRNCLNT_PARAMETER_SPI_CTRL_REQ,                /**< SPI control request (read only)                      */
  VRNCLNT_PARAMETER_VARAN_PORT_AVAIL,            /**< VARAN Port Available Register (read only)            */
  VRNCLNT_PARAMETER_VARAN_PORT_STAT,             /**< VARAN Port Status Register (read only)               */
  VRNCLNT_PARAMETER_VARAN_PORT_CTRL,             /**< VARAN Port Control Register (read only)              */
  VRNCLNT_PARAMETER_FOR_PCK_PORT_AVAIL,          /**< Foreign Package Port Avail Register (read only)      */
  VRNCLNT_PARAMETER_FOR_PCK_PORT_STAT,           /**< Foreign Package Port Status Register (read only)     */
  VRNCLNT_PARAMETER_FOR_PCK_PORT_CTRL,           /**< Foreign Package Port Control Register (read only)    */
  VRNCLNT_PARAMETER_SYSTEM_PERIOD_TIME,          /**< System period time (read only)                       */
  VRNCLNT_PARAMETER_PLL_MAX_VAL_MISS_PLL_SYNC,   /**< Max. Value of allowed missing PLL Input Sync (read-only) */
  VRNCLNT_PARAMETER_DEVICE_DELAY,                /**< Device Delay Time in 10 ns (read only) */
  VRNCLNT_PARAMETER_SYNC_OUT_0_CFG_REG0,         /**< Sync out 0 Offset (read only) */
  VRNCLNT_PARAMETER_SYNC_OUT_0_CFG_REG1,         /**< Sync out 0 Configuration, [7..0]: Period multiplier, [15..8]: Frame count threshold, [23..16]: Sync out ID, [31..24]: Ctrl/Stat */
  VRNCLNT_PARAMETER_SYNC_OUT_1_CFG_REG0,         /**< Sync out 1 Offset (read only) */
  VRNCLNT_PARAMETER_SYNC_OUT_1_CFG_REG1,         /**< Sync out 1 Configuration, [7..0]: Period multiplier, [15..8]: Frame count threshold, [23..16]: Sync out ID, [31..24]: Ctrl/Stat */
  VRNCLNT_PARAMETER_CUR_DIFF,                    /**< current clock count difference between manager and client steps */
  VRNCLNT_PARAMETER_MAX_DIFF,                    /**< maximum clock count difference between manager and client steps */
  VRNCLNT_PARAMETER_ADDED_DIFF,                  /**< added clock count difference between manager and client steps */
  VRNCLNT_PARAMETER_PLL_CLIENT_STEPS,            /**< number of client steps within last sync period */
  VRNCLNT_PARAMETER_PLL_MANAGER_STEPS,           /**< number of manager steps within last sync period */
  VRNCLNT_PARAMETER_CLK_CNT_VAL,                 /**< current clock count value */
  VRNCLNT_PARAMETER_PLL_LOCKED_STAT,             /**< current PLL locked status 1/0: locked/unlocked (read-only) */
  VRNCLNT_PARAMETER_PLL_FRAME_CNT                /**< current PLL frame counter (read-only) */
} VRNCLNT_PARAMETER;

/* Function prototypes */
VRNCLNT_RESULT VaranClient_Init( VRNCLNT_INIT_PARAMETER_SET* ptParams, void* pvUser );
VRNCLNT_RESULT VaranClient_Start( void* pvUser );
VRNCLNT_RESULT VaranClient_GetCounters( VRNCP_COUNTERS_T* ptCntsProcPort, VRNCS_COUNTERS_T* ptCntsSyncPort );
VRNCLNT_RESULT VaranClient_GetIrq( unsigned int uPortNum, uint32_t *pulIrq );
VRNCLNT_RESULT VaranClient_GetMsyncIrq( uint32_t *pulIrq );
VRNCLNT_RESULT VaranClient_ConfirmIrq( unsigned int uPortNum, uint32_t ulConfirmIrqMask );
VRNCLNT_RESULT VaranClient_ReadOutputMemory( unsigned int uiMemIdx, uint32_t * pulOutDataBuf, unsigned int uByteSize );
VRNCLNT_RESULT VaranClient_WriteInputMemory( unsigned int uiMemIdx, uint32_t * pulInDataBuf, unsigned int uByteSize );
VRNCLNT_RESULT VaranClient_ReadSpiBuffer( uint32_t * pulSpiDataBuf );
VRNCLNT_RESULT VaranClient_WriteSpiBuffer( uint32_t * pulSpiData );
VRNCLNT_RESULT VaranClient_ConfirmSpiReq( void );
VRNCLNT_RESULT VaranClient_ConfirmPllUnlocked( void );
VRNCLNT_RESULT VaranClient_SetParameter( uint32_t ulPrmID, uint32_t ulPrmVal );
VRNCLNT_RESULT VaranClient_GetParameter( uint32_t ulPrmID, uint32_t* pulPrmVal );
VRNCLNT_RESULT VaranClient_RecvEthFrame( ETHERNET_FRAME_T* ptFrame );
VRNCLNT_RESULT VaranClient_SendEthFrame( ETHERNET_FRAME_T* ptFrame, int fPrefixPreamble );

#endif /* _HAL_VARAN_CLIENT_H */

